extensions [ gis ]

globals [
  Year
  patch_list
  patch_num
  speed
  neighbor_count
]

patches-own [
  residential_attractiveness
  landcover
  res_dev_prob
  new_residential
  tmp
]

to initialize-global-variables
  set Year 0
  set patch_num 0
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
to initialize [ area ]
  ;; (for this model to work with NetLogo's new plotting features,
  ;; __clear-all-and-reset-ticks should be replaced with clear-all at
  ;; the beginning of your setup procedure and reset-ticks at the end
  ;; of the procedure.)
  __clear-all-and-reset-ticks
  initialize-global-variables
  initialize-maps area
  ask patches [ set new_residential 0 ]

  initialize-edge-patches ; Ring the map with a no-mans-land
  do-plot
  set map-display "Landcover"
end

to initialize-maps [ area ]
  ;; Read these from maps
  let dataset gis:load-dataset (word "maps/" area "/res_attr.asc")
  gis:set-transformation (gis:envelope-of dataset) (list min-pxcor max-pxcor min-pycor max-pycor)  ; Establish transformation
  type "For best results, set max-pxcor to: " 
  type gis:width-of dataset - 1
  type " and max-pycor to: "
  print gis:height-of dataset - 1

  set map-display "Urban Attractiveness"  show-map
  gis:apply-raster dataset residential_attractiveness
  gis:apply-raster gis:load-dataset (word "maps/" area "/landcover.asc") landcover
  set map-display "Landcover"  show-map
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to initialize-edge-patches
  ask patches with [ pxcor = min-pxcor OR pxcor = max-pxcor OR pycor = min-pycor OR pycor = max-pycor ]
  [ set res_dev_prob 0 ]
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to step
  set Year Year + 1
  set map-display "New Development"

  prepare-patch-attractiveness

  let i 0

  while [ i < res_per_year ]
  [
    let j 0
    let l length patch_list
    while [ i < res_per_year AND patch_num < l ]
    [
      ifelse (neighborhoods)
      [
        set neighbor_count 0
        ask ( item patch_num patch_list ) [ grow_neighborhoods neighborhood_size ]
      ]
      [
        ask ( item patch_num patch_list )
        [ 
          if ( developable )
          [
            set neighbor_count 1
            set new_residential 1
            ; Increase attractiveness to new residential
            ;ask neighbors [ set residential_attractiveness residential_attractiveness + ( 1.0 - residential_attractiveness ) * neighbor_effect ]
          ]
        ]
      ]
      set i i + neighbor_count
      set patch_num patch_num + 1
    ]
  ]
  show-map
  do-plot
end

to prepare-patch-attractiveness
  ask patches [
    ifelse ( developable )
    [
      set res_dev_prob random_factor * ( random-float 1.0 ) + (1.0 - random_factor) * residential_attractiveness
    ]
    [ set res_dev_prob 0 ]
  ]  
  set patch_list patches with [res_dev_prob > 0 ]
  set patch_list sort-by [[res_dev_prob] of ?1 > [res_dev_prob] of ?2] patch_list
end

to grow_neighborhoods [ total ]
  if ( developable )
  [ 
    set new_residential 1
    ; Increase attractiveness to new residential
    ;ask neighbors [ set residential_attractiveness residential_attractiveness + ( 1.0 - residential_attractiveness ) * neighbor_effect ]
    set neighbor_count neighbor_count + 1
    if neighbor_count < total
    [ ask neighbors [ grow_neighborhoods total ] ]
  ]
end

to-report developable
  report landcover > 30 AND landcover < 85 AND new_residential = 0
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
; Plot procedures
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to do-plot
  set-current-plot "New Residential"
  set-current-plot-pen "New Lots" plot ( count patches with [ new_residential = 1 ] )
;  set-current-plot-pen "Juveniles" plot ( sum [  Juveniles  ] of patches )
;  set-current-plot-pen "Adults" plot ( sum [ Adults ] of patches )
end

to setup-plot ;; set up plotting
  set-current-plot "New Residential"
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
; Utility procedures
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to show-map
  if ( map-display = "Urban Attractiveness" ) [ ask patches [ set tmp Residential_attractiveness ] show-int-map ]
  if ( map-display = "Landcover" )            [ ask patches [ set tmp LandCover ] show-NLCD-map ]
  
  if ( map-display = "New Development" )      [ ask patches with [ new_residential = 1 ] [ set pcolor red ] ]
end

to show-int-map
  let the_max 0
  set the_max max [ tmp ] of patches
  ifelse ( the_max != 0 )
  [ ask patches 
    [ set pcolor (9.99 - 9.99 * ( tmp / the_max ) ) ]
  ]
  [ ask patches [ set pcolor 0 ] ] 
end

to show-NLCD-map
  ask patches
  [
    ifelse ( tmp = 11 ) [ set pcolor rgb 110 127 177 ] [
    ifelse ( tmp = 21 ) [ set pcolor rgb 253 229 228 ] [
    ifelse ( tmp = 22 ) [ set pcolor rgb 247 178 159 ] [
    ifelse ( tmp = 23 ) [ set pcolor rgb 229 110 110 ] [
    ifelse ( tmp = 31 ) [ set pcolor rgb 210 205 192 ] [
    ifelse ( tmp = 32 ) [ set pcolor rgb 175 175 177 ] [
    ifelse ( tmp = 33 ) [ set pcolor rgb 83 62 118 ] [
    ifelse ( tmp = 41 ) [ set pcolor rgb 133 199 126 ] [
    ifelse ( tmp = 42 ) [ set pcolor rgb 56 129 78 ] [
    ifelse ( tmp = 43 ) [ set pcolor rgb 212 231 176 ] [
    ifelse ( tmp = 51 ) [ set pcolor rgb 220 202 143 ] [
    ifelse ( tmp = 61 ) [ set pcolor rgb 187 174 118 ] [
    ifelse ( tmp = 71 ) [ set pcolor rgb 253 233 170 ] [
    ifelse ( tmp = 81 ) [ set pcolor rgb 251 246 93 ] [
    ifelse ( tmp = 82 ) [ set pcolor rgb 202 145 70 ] [
    ifelse ( tmp = 83 ) [ set pcolor rgb 121 108 74 ] [
    ifelse ( tmp = 84 ) [ set pcolor rgb 244 238 202 ] [
    ifelse ( tmp = 85 ) [ set pcolor rgb 240 156 54 ] [
    ifelse ( tmp = 91 ) [ set pcolor rgb 200 230 248 ] [
    ifelse ( tmp = 92 ) [ set pcolor rgb 100 179 213 ] [ set pcolor black ]
    ]]]]]]]]]]]]]]]]]]]
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
303
10
968
704
-1
-1
1.0
1
10
1
1
1
0
0
0
1
0
654
0
662
0
0
1
ticks
30.0

BUTTON
3
27
147
60
initialize-champaign
initialize \"champaign\"\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

CHOOSER
4
350
186
395
map-display
map-display
"Urban Attractiveness" "Landcover" "New Development"
2

BUTTON
192
350
289
395
NIL
show-map
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
5
256
68
289
NIL
step
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
72
256
135
289
run steps
let i 0\nwhile [ i < years ] [ step   set i i + 1 ]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
249
596
299
641
Year
Year
0
1
11

PLOT
5
441
300
591
New Residential
Time
30x30m lots
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"New Lots" 1.0 0 -2674135 true "" ""

SLIDER
161
256
301
289
years
years
1
100
10
1
1
years
HORIZONTAL

TEXTBOX
5
10
242
28
Step 1: Initialize chosen location\n
11
0.0
1

TEXTBOX
5
239
289
257
Step 3: Choose a number of steps to run, then run
11
0.0
1

TEXTBOX
3
317
294
345
At any time, choose a map to display.  Will display on clicking \"show_map\" and at the end of each step.
11
0.0
1

SLIDER
4
102
147
135
res_per_year
res_per_year
0
10000
1953
1
1
NIL
HORIZONTAL

SWITCH
152
158
297
191
neighborhoods
neighborhoods
1
1
-1000

SLIDER
151
121
297
154
neighborhood_size
neighborhood_size
1
1000
601
10
1
NIL
HORIZONTAL

BUTTON
157
27
291
60
initialize-chapel-hill
initialize \"chapel_hill\"
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
6
85
156
103
Step 2: Set variables
11
0.0
1

SLIDER
5
175
148
208
random_factor
random_factor
0
1
0.23
.01
1
NIL
HORIZONTAL

SLIDER
5
139
148
172
neighbor_effect
neighbor_effect
0
1
0.2
.05
1
NIL
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?

This is a simple urban residential growth model that provides you with playground for understanding urban growth dynamics and a basis for developing more comprehensive models.

## HOW DOES IT WORK?

This model uses two input maps.  One is an attractiveness map for driving development and the other is a National Land Cover Data (NLCD) map used as a background. The attractiveness map was developed with a GRASS GIS script that generates an overall attractiveness to residential development value for every location based on the residential pattern in the NLCD map and shortest travel times to things such as urban centers, highways, roads, interstates, trees, and water.  It also considers slope.  This model is generic in the sense that the attractiveness map can provide attractiveness values for anything. 

## HOW TO USE IT

Maps are provided for Champaign, Ill. and Chapel Hill, N.C.  Begin by initializing the model for one of these locations.  Then, select a "res_per_year" value to reflect the number of 30x30 meter locations to develop into residential.  Optionally choose to turn "neighborhoods" on, which forces development to be clumped. Then run the model for as many steps (each a year) as you choose.

## EXPERIMENTS

Run the model with neighborhoods turned off.  Where is development occurring?  In what ways does this development "make sense"?  In what ways might it be "wrong"?

Play with the neighborhood option turned on.  Can you get the model to be more realistic?  Why or why not?

## MODIFICATIONS

Areas are chosen for development based on 1) the attractiveness map (with values in the range of 0-1.0) and 2) on two random numbers.  The first random number (range 0-1.0) is compared with the attractiveness value.  If it is lower, the location is selected for possible development and is assigned a second random number.  Chosen locations are sorted by this second random number and developed in this order.  When the list is exhausted, the process repeats.

For fun consider adding the ability to simply develop the optimal (most attractive) areas first.  Why might this not match "reality"?  Consider adding a slider that moves between 0 (optimal) and 1 (totally random).  What math might accomplish this?

Development attracts development.  How might the current model be modified to increase the attractiveness of model generated land around new development?

## EXTENSIONS

Development of the attractiveness map considered the urban patterns as a whole.  Consider building different attractiveness maps.  For example, low cost housing, shopping, industrial, and expensive low-density residential.  Consider having these compete with one another.

## CREDITS AND REFERENCES

This model captures notions developed in the creation of the LEAM, SLEUTH, and rSIM models.   This version was developed by Jim Westervelt of the Army Corps' Engineer Research and Development Center.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

link
true
0
Line -7500403 true 150 0 150 300

link direction
true
0
Line -7500403 true 150 150 30 225
Line -7500403 true 150 150 270 225

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0RC3
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
